/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <time.h>
#include <sys/time.h>

#include <algorithm>
#include <list>

#include "Log.h"
#include "CalibrationTypes.h"
#include "Subdivision.h"
#include "WindowSystems/InputEventDispatcher.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"

using namespace std;
using namespace LayerManagerCalibration;
using namespace InputEventProcessing;

InputEventDispatcher::InputEventDispatcher()
    : m_pListener(NULL)
{
}

InputEventDispatcher::~InputEventDispatcher()
{
}

InputEventDispatcher::Listener::Listener()
{
}

InputEventDispatcher::Listener::~Listener()
{
}

InputEventDispatcher::State::State()
{
}

InputEventDispatcher::State::~State()
{
}

bool InputEventDispatcher::State::getSubdivisionPressed(
        evdev_input_device* device, Subdivision* subdivision)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName());

    DeviceSubdivisionPair entry(device, subdivision);

    DeviceSubdivisionMap::iterator match = m_pressedSubdivisions.find(entry);

    bool pressed = (match != m_pressedSubdivisions.end());
    LOG_DEBUG("InputEventDispatcher",
              "  device state=" << (pressed ? "pressed" : "not pressed"));
    return pressed;
}

bool InputEventDispatcher::State::getSubdivisionSlotPressed(
        evdev_input_device* device, Subdivision* subdivision,
        uint slot)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName() << ", slot no.="
              << slot);

    DeviceSubdivisionPair entry(device, subdivision);
    DeviceSubdivisionMap::iterator match = m_pressedSubdivisions.find(entry);
    bool pressed = false;
    if (match != m_pressedSubdivisions.end())
    {
        map<uint,bool>::iterator slot_iter = match->second.find(slot);
        if (slot_iter != match->second.end())
        {
            pressed = true;
        }
    }
    LOG_DEBUG("InputEventDispatcher",
              "slot state=" << (pressed ? "pressed" : "not pressed"));
    return pressed;
}

void InputEventDispatcher::State::setSubdivisionSlotPressed(
        evdev_input_device* device, Subdivision* subdivision,
        uint slot, bool pressed)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName() << ", slot no.="
              << slot << ", pressed state="
              << pressed);

    DeviceSubdivisionPair entry(device, subdivision);
    DeviceSubdivisionMap::iterator match = m_pressedSubdivisions.find(entry);

    if (pressed)
    {
        LOG_DEBUG("InputEventDispatcher",
                  "Mark subdivision as pressed");
        if (match != m_pressedSubdivisions.end())
        {
            LOG_DEBUG("InputEventDispatcher", "Device/Subdivision pair found");

            // Search the entry's list for the slot
            map<uint,bool>::iterator slot_iter = match->second.find(slot);
            if (slot_iter != match->second.end())
            {
                LOG_DEBUG("InputEventDispatcher",
                          "Slot, no.=" << slot
                          << " state is pressed, do nothing");
            }
            else
            {
                LOG_DEBUG("InputEventDispatcher",
                          "Adding slot, no.=" << slot);
                match->second[slot] = true;
            }
        }
        else
        {
            LOG_DEBUG("InputEventDispatcher",
                      "Device/Subdivision pair not found");

            // Create the entry and add the slot to its list.
            LOG_DEBUG("InputEventDispatcher", "Adding slot, no.=" << slot);
            // Create the entry
            m_pressedSubdivisions[entry];
            // Add the slot
            m_pressedSubdivisions[entry].insert(std::make_pair(slot,true));
        }
    }
    else
    {
        if (match != m_pressedSubdivisions.end())
        {
            LOG_DEBUG("InputEventDispatcher",
                      "Device/Subdivision pair found");

            // Search the entry's list for the slot
            map<uint,bool>::iterator slot_iter = match->second.find(slot);

            if (slot_iter != match->second.end())
            {
                LOG_DEBUG("InputEventDispatcher",
                          "Slot is pressed, Removing slot, no.=" << slot);
                match->second.erase(slot_iter);
            }

            if (match->second.empty())
            {
                LOG_DEBUG("InputEventDispatcher",
                          "Entry is empty, removing it");
                m_pressedSubdivisions.erase(entry);
            }
        }
        else
        {
            LOG_DEBUG("InputEventDispatcher",
                      "Device/Subdivision pair not found,"
                      " already unpressed");
        }
    }
}

bool InputEventDispatcher::State::getTimeOfLastActivation(
        struct evdev_input_device* device,
        LayerManagerCalibration::Subdivision* subdivision,
        uint& time)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName());

    DeviceSubdivisionPair entry(device, subdivision);
    DeviceSubdivisonTimeMap::iterator match =
            m_timeOfSubdivisionLastActivation.find(entry);
    bool matchFound = false;

    if (match != m_timeOfSubdivisionLastActivation.end())
    {
        time = match->second;
        matchFound = true;
    }
    else
    {
        LOG_DEBUG("InputEventDispatcher",
                  "Device/Subdivision pair not found");
    }

    return matchFound;
}

void InputEventDispatcher::State::setTimeOfLastActivation(
        struct evdev_input_device* device,
        LayerManagerCalibration::Subdivision* subdivision,
        uint timeStamp)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName() << ", time stamp="
              << timeStamp);

    DeviceSubdivisionPair entry(device, subdivision);
    DeviceSubdivisonTimeMap::iterator match =
            m_timeOfSubdivisionLastActivation.find(entry);

    if (match != m_timeOfSubdivisionLastActivation.end())
    {
        LOG_DEBUG("InputEventDispatcher",
                  "Device/Subdivision pair found set time stamp");
        m_timeOfSubdivisionLastActivation[entry] = timeStamp;
    }
    else
    {
        LOG_DEBUG("InputEventDispatcher",
                  "Device/Subdivision pair not found");

        // Create the entry and add the slot to its list.
        LOG_DEBUG("InputEventDispatcher",
                  "Adding subdivision, name=" << subdivision->getName()
                  << " set last activation time="
                  << timeStamp);
        // Create the entry
        m_timeOfSubdivisionLastActivation[entry];
        // Add entry
        m_timeOfSubdivisionLastActivation[entry] = timeStamp;
    }
}

InputEventDispatcher::State& InputEventDispatcher::getState()
{
    return m_state;
}

void InputEventDispatcher::setListener(InputEventDispatcher::Listener* listener)
{
    m_pListener = listener;
}

InputEventDispatcher::Listener* InputEventDispatcher::getListener() const
{
    return m_pListener;
}

uint InputEventDispatcher::getCurrentTime() const
{
#ifdef WL_OMIT_GETTIME
    return 0xF0F0F0F0;
#else
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec * 1000 + tv.tv_usec / 1000;
#endif
}

bool InputEventDispatcher::State::getSubdivisionSlotMap(struct evdev_input_device* device,
                                                        LayerManagerCalibration::Subdivision* subdivision,
                                                        std::map<uint,bool>** slotMap)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName());

    bool slotsFound = false;
    DeviceSubdivisionPair entry(device, subdivision);
    DeviceSubdivisionMap::iterator match = m_pressedSubdivisions.find(entry);
    if (match != m_pressedSubdivisions.end())
    {
        LOG_DEBUG("SliderInputEventEventDispatcher",
                  "Found slots entries for subdivision");
        *slotMap = &(match->second);
        slotsFound = true;
    }

    return slotsFound;
}

void InputEventDispatcher::State::setSubdivisionSlotValidity(struct evdev_input_device* device,
                                                             Subdivision* subdivision,
                                                             uint slot,
                                                             bool validity)
{
    LOG_DEBUG("InputEventDispatcher",
              "device name="
              << device->deviceName << ", subdivision name="
              << subdivision->getName() << ", slot no.="
              << slot << ", validity="
              << validity);

    DeviceSubdivisionPair entry(device, subdivision);
    DeviceSubdivisionMap::iterator match = m_pressedSubdivisions.find(entry);


        LOG_DEBUG("InputEventDispatcher", "Mark subdivision as pressed");
        if (match != m_pressedSubdivisions.end())
        {
            LOG_DEBUG("InputEventDispatcher", 
                      "Device/Subdivision pair found");

            // Search the entry's list for the slot
            map<uint,bool>::iterator slot_iter = match->second.find(slot);
            if (slot_iter != match->second.end())
            {
                if (slot_iter->second == validity)
                {
                    LOG_DEBUG("InputEventDispatcher", "Slot, no.=" << slot
                              << " already pressed, and validity set"
                              << ", doing nothing");
                }
                else
                {
                    LOG_DEBUG("InputEventDispatcher", "Slot, no.=" << slot
                              << " already pressed, set validity");
                    match->second[slot] = validity;
                }
            }
            else
            {
                LOG_DEBUG("InputEventDispatcher", "Adding slot, no.="
                          << slot << ", and validity "
                          << validity);

                match->second[slot] = validity;
            }
        }
        else
        {
            LOG_DEBUG("InputEventDispatcher",
                      "Device/Subdivision pair not found");
        }

}
